import { PathFn, TimingFn } from 'mutate-animate';
import { Polygon } from './polygon';
interface TransitionInfo {
    time: number;
    mode: TimingFn;
}
export interface Light {
    id: string;
    x: number;
    y: number;
    r: number;
    /** 衰减开始半径 */
    decay: number;
    /** 颜色，每个值的范围0.0~1.0 */
    color: Color;
    /** 是否可以被物体遮挡 */
    noShelter?: boolean;
    /** 是否跟随勇士 */
    followHero?: boolean;
    /** 正在动画的属性 */
    _animating?: Record<string, boolean>;
    /** 执行渐变的属性 */
    _transition?: Record<string, TransitionInfo>;
    /** 表示是否是代理，只有设置渐变后才会变为true */
    _isProxy?: boolean;
    /** 跟随勇士的时候的偏移量 */
    _offset?: Loc;
}
export declare function init(): void;
/**
 * 初始化阴影画布
 */
export declare function initShadowCanvas(): void;
/**
 * 添加一个光源
 * @param info 光源信息
 */
export declare function addLight(info: Light): void;
/**
 * 移除一个光源
 * @param id 光源id
 */
export declare function removeLight(id: string): void;
/**
 * 设置一个光源的信息
 * @param id 光源id
 * @param info 光源信息
 */
export declare function setLight(id: string, info: Partial<Light>): void;
/**
 * 设置当前的光源列表
 * @param list 光源列表
 */
export declare function setLightList(list: Light[]): void;
/**
 * 去除所有的光源
 */
export declare function removeAllLights(): void;
/**
 * 获取一个灯光
 * @param id 灯光id
 */
export declare function getLight(id: string): Light | undefined;
/**
 * 获取所有灯光
 */
export declare function getAllLights(): Light[];
/**
 * 设置背景色
 * @param color 背景色
 */
export declare function setBackground(color: Color): void;
/**
 * 刷新灯光信息并重绘
 */
export declare function refreshLight(): void;
/**
 * 动画改变一个属性的值
 * @param id 灯光id
 * @param key 动画属性，x,y,r,decay，颜色请使用animateLightColor（下个版本会加）
 * @param n 目标值
 * @param time 动画时间
 * @param mode 动画方式，渐变函数，高级动画提供了大量内置的渐变函数
 * @param relative 相对方式，是绝对还是相对
 */
export declare function animateLight<K extends Exclude<keyof Light, 'id'>>(id: string, key: K, n: Light[K], time?: number, mode?: TimingFn, relative?: boolean): void;
/**
 * 把一个属性设置为渐变模式
 * @param id 灯光id
 * @param key 渐变的属性
 * @param time 渐变时长
 * @param mode 渐变方式，渐变函数，高级动画提供了大量内置的渐变函数
 */
export declare function transitionLight<K extends Exclude<keyof Light, 'id'>>(id: string, key: K, time?: number, mode?: TimingFn): void;
/**
 * 移动一个灯光
 * @param id 灯光id
 * @param x 目标横坐标
 * @param y 目标纵坐标
 * @param time 移动时间
 * @param mode 移动方式，渐变函数
 * @param relative 相对模式，相对还是绝对
 */
export declare function moveLight(id: string, x: number, y: number, time?: number, mode?: TimingFn, relative?: boolean): void;
/**
 * 以一个路径移动光源
 * @param id 灯光id
 * @param time 移动时长
 * @param path 移动路径
 * @param mode 移动方式，渐变函数，表示移动的完成度
 * @param relative 相对模式，相对还是绝对
 */
export declare function moveLightAs(id: string, time: number, path: PathFn, mode?: TimingFn, relative?: boolean): void;
export declare function animateLightColor(id: string, target: Color, time?: number, mode?: TimingFn): void;
/**
 * 根据坐标数组设置物体节点
 * @param nodes 坐标数组
 */
export declare function setShadowNodes(nodes: LocArr[][]): void;
/**
 * 根据多边形数组设置物体节点
 * @param nodes 多边形数组
 */
export declare function setShadowNodes(nodes: Polygon[]): void;
/**
 * 根据坐标数组添加物体节点
 * @param polygons 坐标数组
 */
export declare function addPolygon(...polygons: LocArr[][]): void;
/**
 * 根据多边形数组添加物体节点
 * @param polygons 多边形数组
 */
export declare function addPolygon(...polygons: Polygon[]): void;
/**
 * 设置光源的虚化程度
 * @param n 虚化程度
 */
export declare function setBlur(n: number): void;
/**
 * 绘制阴影
 */
export declare function drawShadow(): void;
export {};
