import { FunctionalComponent } from 'vue';
import { EmitableEvent, EventEmitter } from '../common/eventEmitter';
import { GameStorage } from './storage';
export interface SettingComponentProps {
    item: MotaSettingItem;
    setting: MotaSetting;
    displayer: SettingDisplayer;
}
export type SettingComponent = FunctionalComponent<SettingComponentProps>;
type MotaSettingType = boolean | number | MotaSetting;
export interface MotaSettingItem<T extends MotaSettingType = MotaSettingType> {
    name: string;
    key: string;
    value: T;
    controller: SettingComponent;
    description?: string;
    defaults?: boolean | number;
    step?: [number, number, number];
    display?: (value: T) => string;
}
interface SettingEvent extends EmitableEvent {
    valueChange: <T extends boolean | number>(key: string, newValue: T, oldValue: T) => void;
}
export type SettingText = {
    [key: string]: string[] | SettingText;
};
export interface SettingDisplayInfo {
    item: MotaSettingItem | null;
    list: Record<string, MotaSettingItem>;
    text: string[];
}
export declare class MotaSetting extends EventEmitter<SettingEvent> {
    static noStorage: string[];
    readonly list: Record<string, MotaSettingItem>;
    /**
     * 重设设置
     * @param setting 设置信息
     */
    reset(setting: Record<string, boolean | number>): void;
    /**
     * 注册一个数字型设置
     * @param key 设置的键名
     * @param value 设置的值
     */
    register(key: string, name: string, value: number, com?: SettingComponent, step?: [number, number, number]): this;
    /**
     * 注册一个非数字型设置
     * @param key 设置的键名
     * @param value 设置的值
     */
    register(key: string, name: string, value: boolean | MotaSetting, com?: SettingComponent): this;
    /**
     * 获取一个设置信息
     * @param key 要获取的设置的键
     */
    getSetting(key: string): Readonly<MotaSettingItem | null>;
    /**
     * 设置一个设置的值
     * @param key 要设置的设置的键
     * @param value 要设置的值
     */
    setValue(key: string, value: boolean | number, noEmit?: boolean): void;
    /**
     * 增加一个设置的值
     * @param key 要改变的设置的值
     * @param value 值的增量
     */
    addValue(key: string, value: number): void;
    /**
     * 获取一个设置的值，如果获取到的是一个MotaSetting实例，那么返回undefined
     * @param key 要获取的设置
     */
    getValue(key: string): boolean | number | undefined;
    /**
     * 获取一个设置的值，如果获取到的是一个MotaSetting实例，那么返回defaultValue
     * @param key 要获取的设置
     * @param defaultValue 设置的默认值
     */
    getValue<T extends boolean | number>(key: string, defaultValue: T): T;
    /**
     * 设置一个设置的值显示函数
     * @param key 要设置的设置的键
     * @param func 显示函数
     */
    setDisplayFunc(key: string, func: (value: MotaSettingType) => string): this;
    /**
     * 设置一个设置的修改部分组件
     * @param key 要设置的设置的键
     * @param com 设置修改部分的组件
     */
    setValueController(key: string, com: SettingComponent): this;
    /**
     * 设置一个设置的说明
     * @param key 要设置的设置的id
     * @param desc 设置的说明
     */
    setDescription(key: string, desc: string): this;
    private getSettingBy;
}
interface SettingDisplayerEvent extends EmitableEvent {
    update: (stack: string[], display: SettingDisplayInfo[]) => void;
}
export declare class SettingDisplayer extends EventEmitter<SettingDisplayerEvent> {
    setting: MotaSetting;
    /** 选项选中栈 */
    selectStack: string[];
    displayInfo: SettingDisplayInfo[];
    constructor(setting: MotaSetting);
    /**
     * 添加选择项
     * @param key 下一个选择项
     */
    add(key: string): void;
    /**
     * 剪切后面的选择项
     * @param index 从哪开始剪切
     */
    cut(index: number, noUpdate?: boolean): void;
    update(): void;
}
export declare const mainSetting: MotaSetting;
declare const storage: GameStorage<any>;
export { storage as settingStorage };
