import { EmitableEvent, EventEmitter } from '@/core/common/eventEmitter';
import { KeyCode } from '@/plugin/keyCodes';
import { FunctionalComponent } from 'vue';
interface CustomToolbarEvent extends EmitableEvent {
    add: (item: ValueOf<ToolbarItemMap>) => void;
    delete: (item: ValueOf<ToolbarItemMap>) => void;
    set: (id: string, data: Partial<SettableItemData>) => void;
    emit: (id: string, item: ValueOf<ToolbarItemMap>) => void;
    posChange: (bar: CustomToolbar) => void;
}
interface ToolbarItemBase<T extends ToolbarItemType> {
    type: T;
    id: string;
}
interface HotkeyToolbarItem extends ToolbarItemBase<'hotkey'> {
    key: KeyCode;
    assist: number;
}
interface ItemToolbarItem extends ToolbarItemBase<'item'> {
    item: ItemIdOf<'tools' | 'constants'>;
}
interface AssistKeyToolbarItem extends ToolbarItemBase<'assistKey'> {
    assist: KeyCode.Ctrl | KeyCode.Shift | KeyCode.Alt;
}
interface ToolbarItemMap {
    hotkey: HotkeyToolbarItem;
    item: ItemToolbarItem;
    assistKey: AssistKeyToolbarItem;
}
export type ToolbarItemType = keyof ToolbarItemMap;
export type SettableItemData<T extends ToolbarItemType = ToolbarItemType> = Omit<ToolbarItemMap[T], 'id' | 'type'>;
export interface CustomToolbarProps<T extends ToolbarItemType = ToolbarItemType> {
    item: ToolbarItemMap[T];
    toolbar: CustomToolbar;
}
export type CustomToolbarComponent<T extends ToolbarItemType = ToolbarItemType> = FunctionalComponent<CustomToolbarProps<T>>;
type ToolItemEmitFn<T extends ToolbarItemType> = (this: CustomToolbar, id: string, item: ToolbarItemMap[T]) => boolean;
interface RegisteredCustomToolInfo {
    name: string;
    onEmit: ToolItemEmitFn<ToolbarItemType>;
    show: CustomToolbarComponent;
    editor: CustomToolbarComponent;
    onCreate: (item: any) => ToolbarItemBase<ToolbarItemType>;
}
export declare class CustomToolbar extends EventEmitter<CustomToolbarEvent> {
    static num: number;
    static list: CustomToolbar[];
    static info: Record<string, RegisteredCustomToolInfo>;
    items: ValueOf<ToolbarItemMap>[];
    num: number;
    id: string;
    x: number;
    y: number;
    width: number;
    height: number;
    assistKey: number;
    showIds: number[];
    constructor(id: string);
    /**
     * 添加一个自定义项
     * @param item 要添加的自定义工具栏项
     */
    add<K extends ToolbarItemType>(item: ToolbarItemMap[K]): this;
    /**
     * 删除一个自定义项
     * @param id 要删除的项的id
     */
    delete(id: string): this | undefined;
    /**
     * 设置一个项
     * @param id 要设置的项的id
     * @param item 要设置的属性内容
     */
    set<T extends ToolbarItemType>(id: string, item: Partial<SettableItemData<T>>): this | undefined;
    /**
     * 触发一个自定义工具
     * @param id 要触发的自定义工具的id
     */
    emitTool(id: string): this;
    /**
     * 强制刷新这个自定义工具栏的所有显示
     */
    refresh(reopen?: boolean): this;
    setPos(x?: number, y?: number): void;
    setSize(width?: number, height?: number): void;
    /**
     * 显示这个自定义工具栏，可以显示多个，且内容互通
     */
    show(): number;
    /**
     * 关闭一个以此实例为基础显示的自定义工具栏
     * @param id 要关闭的id
     */
    close(id: number): void;
    /**
     * 关闭这个自定义工具栏的所有显示
     */
    closeAll(): void;
    static get(id: string): CustomToolbar | undefined;
    /**
     * 注册一类自定义工具
     * @param type 要注册的自定义工具类型
     * @param name 该类型的中文名
     * @param onEmit 当触发这个自定义工具的时候执行的函数
     * @param show 这个自定义工具在自定义工具栏的显示组件
     * @param editor 这个自定义工具在编辑时编辑组件
     * @param onCreate 当这个自定义工具在编辑器中被添加时，执行的初始化脚本
     */
    static register<K extends ToolbarItemType>(type: K, name: string, onEmit: ToolItemEmitFn<K>, show: CustomToolbarComponent<K>, editor: CustomToolbarComponent<K>, onCreate: (item: any) => ToolbarItemMap[K]): void;
    static save(): void;
    static load(): void;
    static refreshAll(reopen?: boolean): void;
    static showAll(): number[];
    static closeAll(): void;
}
export {};
