import { AxiosResponse } from 'axios';
import { Disposable } from '../common/disposable';
import JSZip from 'jszip';
import { EmitableEvent, EventEmitter } from '../common/eventEmitter';
interface ResourceData {
    image: HTMLImageElement;
    arraybuffer: ArrayBuffer;
    text: string;
    json: any;
    zip: ZippedResource;
    bgm: HTMLAudioElement;
}
export type ResourceType = keyof ResourceData;
export type NonZipResource = Exclude<ResourceType, 'zip'>;
export declare class Resource<T extends ResourceType = ResourceType> extends Disposable<string> {
    format: T;
    request?: Promise<AxiosResponse<ResourceData[T]> | '@imageLoaded' | '@bgmLoaded'>;
    loaded: boolean;
    uri: string;
    type: string;
    name: string;
    ext: string;
    /** 资源数据 */
    resource?: ResourceData[T];
    constructor(resource: string, format: T);
    protected onLoadStart(v?: ResourceData[T]): void;
    protected onLoad(v: ResourceData[T]): void;
    /**
     * 解析资源url
     * @param resource 资源字符串
     * @returns 解析出的资源url
     */
    protected resolveUrl(resource: string): string;
    /**
     * 加载资源
     */
    protected load(): void;
    /**
     * 获取资源，如果还没加载会等待加载完毕再获取
     */
    getData(): Promise<ResourceData[T] | null>;
    /**
     * 设置资源数据，不再需要加载
     * @param data 数据
     */
    protected setData(data: ResourceData[T] | Promise<ResourceData[T]>): this;
}
interface ZippedEvent extends EmitableEvent {
    ready: (data: JSZip) => void;
}
export declare class ZippedResource extends EventEmitter<ZippedEvent> {
    zip: Promise<JSZip>;
    data?: JSZip;
    constructor(buffer: ArrayBuffer);
}
export declare class ResourceStore<T extends ResourceType> extends Map<string, Resource<T>> {
    active(key: string[] | string): void;
    dispose(key: string[] | string): void;
    destroy(key: string[] | string): void;
    push(data: [string, Resource<T>][] | Record<string, Resource<T>>): void;
    getData<T extends ResourceType = ResourceType>(key: string): Promise<ResourceData[T] | null>;
    getDataSync<T extends ResourceType = ResourceType>(key: string): ResourceData[T] | null;
}
export declare function getTypeByResource(resource: string): ResourceType;
export declare function getZipFormatByType(type: ResourceType): 'arraybuffer' | 'text';
export declare const resource: ResourceStore<keyof ResourceData>;
export declare const zipResource: ResourceStore<keyof ResourceData>;
export {};
